implementation module ospicture

//	Version 1.0.2

//
//	Drawing functions and other operations on Pictures. 
//

import clCrossCall
import osfont, ostoolbox
import commondef, StdPictureDef


::	Picture
	=	{	pictContext		:: !OSPictContext	// The context for drawing operations
		,	pictToolbox		:: !.OSToolbox		// The continuation value
		,	pictOrigin		:: !Origin			// The current origin of the picture
		,	pictPen			:: !Pen				// The current state of the pen
		}
::	Origin
	:==	Point
::	OSPictContext
	:==	HDC
::  Pen
	=	{	penSize			:: !Int				// The width and height of the pen
  		,	penForeColour	:: !Colour			// The drawing colour of the pen
		,	penBackColour	:: !Colour			// The background colour of the pen
		,	penPos			:: !Point			// The pen position in local coordinates
		,	penFont			:: !Font			// The font information to draw text and characters
		}


//	Conversion operations to and from Picture
/*
initialisePicture :: !Origin !Pen !OSPictContext !*OSToolbox -> (!OSPictContext,!*OSToolbox)
initialisePicture origin pen=:{penSize,penForeColour,penBackColour,penPos,penFont} hdc tb
	# {osfontname,osfontstyles,osfontsize}	= OSfontgetimp penFont
	# (hdc,tb)								= WinInitPicture
												penSize
												iModeCopy
												initforecolour
												initbackcolour
												initpen
												(osfontname,osfontstyles,osfontsize)
												(0,0)
												(hdc,tb)
	# (_,_,_,_,_,_,(hdc,tb))	= WinDonePicture (hdc,tb)
	= (hdc,tb)
where
	initforecolour	= toRGBtriple penForeColour
	initbackcolour	= toRGBtriple penBackColour
	initpen			= toTuple (penPos-origin)
*/
packPicture :: !Origin !Pen !OSPictContext !*OSToolbox -> *Picture
packPicture origin pen=:{penSize,penForeColour,penBackColour,penPos,penFont} hdc tb
	#! {osfontname,osfontstyles,osfontsize}= OSfontgetimp penFont
	#! (hdc,tb)		= WinInitPicture
						penSize
						iModeCopy
						initforecolour
						initbackcolour
						initpen
						(osfontname,osfontstyles,osfontsize)
						(0,0)
						(hdc,tb)
	= {	pictContext	= hdc
	  ,	pictToolbox	= tb
	  ,	pictOrigin	= origin
	  ,	pictPen		= pen
	  }
where
	initforecolour	= toRGBtriple penForeColour
	initbackcolour	= toRGBtriple penBackColour
	initpen			= toTuple (penPos-origin)

unpackPicture :: !*Picture -> (!Origin,!Pen,!OSPictContext,!*OSToolbox)
unpackPicture {pictOrigin,pictPen,pictContext,pictToolbox}
// PA: intend to use simplified version of WinDonePicture; crashes for some reason.
	# (_,_,_,_,_,_,(hdc,tb))	= WinDonePicture (pictContext,pictToolbox)
//	# (hdc,tb)	= WinDonePicture (pictContext,pictToolbox)
	= (pictOrigin,pictPen,hdc,tb)

peekPicture :: !*Picture -> (!Origin,!Pen,!OSPictContext,!*OSToolbox)
peekPicture {pictOrigin,pictPen,pictContext,pictToolbox}
	= (pictOrigin,pictPen,pictContext,pictToolbox)

unpeekPicture :: !Origin !Pen !OSPictContext !*OSToolbox -> *Picture
unpeekPicture origin pen hdc tb
	= {pictOrigin=origin,pictPen=pen,pictContext=hdc,pictToolbox=tb}

peekOSPictContext :: !*Picture -> (!OSPictContext,!*Picture)
peekOSPictContext picture=:{pictContext}
	= (pictContext,picture)

sharePicture :: !*Picture -> (!Picture,!*Picture)
sharePicture picture=:{pictOrigin,pictPen}
	= ({pictContext=0,pictToolbox=OSNewToolbox,pictOrigin=pictOrigin,pictPen=pictPen},picture)

peekScreen :: !(St *Picture .x) !*OSToolbox -> (!.x,!*OSToolbox)
peekScreen f tb
	# (hdc,tb)		= WinCreateScreenHDC tb
	# picture		= packPicture zero defaultPen hdc tb
	# (x,picture)	= f picture
	# (_,_,hdc,tb)	= unpackPicture picture
	# tb			= WinDestroyScreenHDC (hdc,tb)
	= (x,tb)


defaultPen :: Pen
defaultPen
	= {	penSize			= 1
	  ,	penForeColour	= Black
	  ,	penBackColour	= White
	  ,	penPos			= zero
	  ,	penFont			= defaultFont
	  }
where
	(defaultFont,_)		= OSdefaultfont OSNewToolbox

dialogPen :: Pen
dialogPen
	= {	penSize			= 1
	  ,	penForeColour	= Black
	  ,	penBackColour	= White
	  ,	penPos			= zero
	  ,	penFont			= dialogFont
	  }
where
	(dialogFont,_)		= OSdialogfont OSNewToolbox


/*	Picture interface functions.
*/
apppicttoolbox :: !(IdFun *OSToolbox) !*Picture -> *Picture
apppicttoolbox f picture=:{pictToolbox}
	= {picture & pictToolbox=f pictToolbox}

accpicttoolbox :: !(St *OSToolbox .x) !*Picture -> (!.x,!*Picture)
accpicttoolbox f picture=:{pictToolbox}
	# (x,tb)	= f pictToolbox
	= (x,{picture & pictToolbox=tb})


/*	Attribute functions.
*/
//	Access to Origin and Pen:
getpictorigin :: !*Picture -> (!Origin,!*Picture)
getpictorigin picture=:{pictOrigin}
	= (pictOrigin,picture)

setpictorigin :: !Origin !*Picture -> *Picture
setpictorigin origin picture
	= {picture & pictOrigin=origin}

getpictpen :: !*Picture -> (!Pen,!*Picture)
getpictpen picture=:{pictPen}
	= (pictPen,picture)

setpictpen :: !Pen !*Picture -> *Picture
setpictpen {penSize,penForeColour,penBackColour,penPos,penFont} picture
	# picture	= setpictpensize    penSize       picture
	# picture	= setpictpencolour  penForeColour picture
	# picture	= setpictbackcolour penBackColour picture
	# picture	= setpictpenpos     penPos        picture
	# picture	= setpictpenfont    penFont       picture
	= picture


//	Change the pen position:
setpictpenpos :: !Point !*Picture -> *Picture
setpictpenpos newpos picture=:{pictToolbox,pictOrigin,pictPen,pictContext}
	| newpos==pictPen.penPos
		= picture
	| otherwise
		# (context,tb)	= WinMovePenTo (toTuple (newpos-pictOrigin)) (pictContext,pictToolbox)
		  pen			= {pictPen & penPos=newpos}
		= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

getpictpenpos :: !*Picture -> (!Point,!*Picture)
getpictpenpos picture=:{pictPen={penPos}}
	= (penPos,picture)

movepictpenpos :: !Vector !*Picture -> *Picture
movepictpenpos v picture=:{pictToolbox,pictPen,pictContext}
	# (context,tb)	= WinMovePen (toTuple v) (pictContext,pictToolbox)
	  pen			= {pictPen & penPos=movePoint v pictPen.penPos}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

//	Change the pen size:
setpictpensize :: !Int !*Picture -> *Picture
setpictpensize w picture=:{pictToolbox,pictContext,pictPen}
	| w`==pictPen.penSize
		= picture
	| otherwise
		# (context,tb)	= WinSetPenSize w` (pictContext,pictToolbox)
		  pen			= {pictPen & penSize=w`}
		= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}
where
	w` = max 1 w

getpictpensize :: !*Picture -> (!Int,!*Picture)
getpictpensize picture=:{pictPen={penSize}}
	= (penSize,picture)


//	Change the PenColour:
setpictpencolour :: !Colour !*Picture -> *Picture
setpictpencolour colour picture=:{pictToolbox,pictPen,pictContext}
	| reqRGB==curRGB
		= picture
	| otherwise
		# (context,tb)	= WinSetPenColor reqRGB (pictContext,pictToolbox)
		  pen			= {pictPen & penForeColour=colour}
		= {picture & pictPen=pen,pictToolbox=tb,pictContext=context}
where
	reqRGB				= toRGBtriple colour
	curRGB				= toRGBtriple pictPen.penForeColour

setpictbackcolour :: !Colour !*Picture -> *Picture
setpictbackcolour colour picture=:{pictToolbox,pictPen,pictContext}
	| reqRGB==curRGB
		= picture
	| otherwise
		# (context,tb)	= WinSetBackColor (toRGBtriple colour) (pictContext,pictToolbox)
		  pen			= {pictPen & penBackColour=colour}
		= {picture & pictPen=pen,pictToolbox=tb,pictContext=context}
where
	reqRGB				= toRGBtriple colour
	curRGB				= toRGBtriple pictPen.penBackColour

toRGBtriple :: !Colour -> (!Int,!Int,!Int)
toRGBtriple (RGB {r,g,b})	= (SetBetween r MinRGB MaxRGB,SetBetween g MinRGB MaxRGB,SetBetween b MinRGB MaxRGB)
toRGBtriple Black			= (MinRGB,MinRGB,MinRGB)
toRGBtriple DarkGrey		= ( MaxRGB>>2,    MaxRGB>>2,    MaxRGB>>2)
toRGBtriple Grey			= ( MaxRGB>>1,    MaxRGB>>1,    MaxRGB>>1)
toRGBtriple LightGrey		= ((MaxRGB>>2)*3,(MaxRGB>>2)*3,(MaxRGB>>2)*3)
toRGBtriple White			= (MaxRGB,MaxRGB,MaxRGB)
toRGBtriple Red				= (MaxRGB,MinRGB,MinRGB)
toRGBtriple Green			= (MinRGB,MaxRGB,MinRGB)
toRGBtriple Blue			= (MinRGB,MinRGB,MaxRGB)
toRGBtriple Cyan			= (MinRGB,MaxRGB,MaxRGB)
toRGBtriple Magenta			= (MaxRGB,MinRGB,MaxRGB)
toRGBtriple Yellow			= (MaxRGB,MaxRGB,MinRGB)

getpictpencolour :: !*Picture -> (!Colour,!*Picture)
getpictpencolour picture=:{pictPen={penForeColour}}
	= (penForeColour,picture)

// MW: can be removed in version 1.3
accpictpencolour :: !*Picture -> (!Colour,!*Picture)
accpictpencolour picture=:{pictPen={penForeColour}}
	=(penForeColour,picture)

getpictbackcolour :: !*Picture -> (!Colour,!*Picture)
getpictbackcolour picture=:{pictPen={penBackColour}}
	= (penBackColour,picture)


//	Change the font attributes:
setpictpenfont :: !Font !*Picture -> *Picture
setpictpenfont font picture=:{pictToolbox,pictContext,pictPen=pen}
	| imp==OSfontgetimp pen.penFont
		= picture
	| otherwise
		# (context,tb)	= WinSetFont (osfontname,osfontstyles,osfontsize) (pictContext,pictToolbox)
		  pen			= {pen & penFont=font}
		= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}
where
	imp										= OSfontgetimp font
	{osfontname,osfontstyles,osfontsize}	= imp

getpictpenfont :: !*Picture -> (!Font,!*Picture)
getpictpenfont picture=:{pictPen={penFont}}
	= (penFont,picture)

setpictpendefaultfont :: !*Picture -> *Picture
setpictpendefaultfont picture=:{pictToolbox,pictContext,pictPen}
	# (font,tb)		= OSdefaultfont pictToolbox
	  {osfontname,osfontstyles,osfontsize}
	  				= OSfontgetimp font
	# (context,tb)	= WinSetFont (osfontname,osfontstyles,osfontsize) (pictContext,tb)
	  pen			= {pictPen & penFont=font}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}


/*	Drawing mode setting functions.
*/
setpictxormode :: !*Picture -> *Picture
setpictxormode picture=:{pictToolbox,pictContext}
	# (context,tb)	= WinSetMode iModeXor (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

setpicthilitemode :: !*Picture -> *Picture
setpicthilitemode picture=:{pictToolbox,pictContext}
	# (context,tb)	= WinSetMode iModeXor (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

setpictnormalmode :: !*Picture -> *Picture
setpictnormalmode picture=:{pictToolbox,pictContext}
	# (context,tb)	= WinSetMode iModeCopy (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}


/*	Point drawing operations.
	pictdrawpoint
		only draws a point at that position. The pen position is not changed.
*/
pictdrawpoint :: !Point !*Picture -> *Picture
pictdrawpoint pos=:{x,y} picture=:{pictPen={penSize},pictOrigin={x=ox,y=oy},pictToolbox,pictContext}
	| penSize==1
		# (context,tb)	= WinDrawPoint (x`,y`) (pictContext,pictToolbox)
		= {picture & pictToolbox=tb,pictContext=context}
	| otherwise
		# (context,tb)	= WinFillRectangle (x`,y`,x`+penSize,y`+penSize) (pictContext,pictToolbox)
		= {picture & pictToolbox=tb,pictContext=context}
where
	(x`,y`)	= (x-ox,y-oy)


/*	Line drawing operations.
	pictdrawlineto
		draws a line from the current pen position to the given pen position. 
		The new pen position is the endpoint of the line.	
	pictdrawline
		draws a line from the first point to the second point. The pen position
		is not changed.
*/
pictdrawlineto :: !Point !*Picture -> *Picture
pictdrawlineto pos picture=:{pictOrigin,pictToolbox,pictContext,pictPen}
	# (context,tb)	= WinLinePenTo (toTuple (pos-pictOrigin)) (pictContext,pictToolbox)
	  pen			= {pictPen & penPos=pos}
	= {picture & pictToolbox=tb,pictContext=context,pictPen=pen}

pictundrawlineto :: !Point !*Picture -> *Picture
pictundrawlineto pos picture=:{pictOrigin,pictToolbox,pictContext,pictPen=pen=:{penForeColour,penBackColour}}
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= WinLinePenTo (toTuple (pos-pictOrigin)) (context,tb)
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictToolbox=tb,pictContext=context,pictPen={pen & penPos=pos}}

pictdrawline :: !Point !Point !*Picture -> *Picture
pictdrawline a b picture=:{pictOrigin,pictToolbox,pictContext}
	# (context,tb)	= WinDrawLine (toTuple (a-pictOrigin)) (toTuple (b-pictOrigin)) (pictContext,pictToolbox)
	= {picture & pictToolbox=tb,pictContext=context}

pictundrawline :: !Point !Point !*Picture -> *Picture
pictundrawline a b picture=:{pictOrigin,pictToolbox,pictContext,pictPen={penForeColour,penBackColour}}
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= WinDrawLine (toTuple (a-pictOrigin)) (toTuple (b-pictOrigin)) (context,tb)
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictToolbox=tb,pictContext=context}


/*	Text drawing operations.
	pictdraw(char/string) draws a char/string at the current pen position. The new
		pen position is immediately after the drawn char/string.
*/
pictdrawchar :: !Char !*Picture -> *Picture
pictdrawchar char picture=:{pictContext,pictToolbox,pictPen,pictOrigin={x=ox,y=oy}}
	# (context,tb)			= WinDrawChar (toInt char) (pictContext,pictToolbox)
	# ((x,y),(context,tb))	= WinGetPenPos (context,tb)
	  pen					= {pictPen & penPos={x=x+ox,y=y+oy}}
	= {picture & pictContext=context,pictToolbox=tb,pictPen=pen}

pictundrawchar :: !Char !*Picture -> *Picture
pictundrawchar char picture=:{pictContext,pictToolbox,pictPen=pen=:{penForeColour,penBackColour},pictOrigin={x=ox,y=oy}}
	# (context,tb)			= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)			= WinDrawChar (toInt char) (context,tb)
	# (context,tb)			= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	# ((x,y),(context,tb))	= WinGetPenPos (context,tb)
	= {picture & pictContext=context,pictToolbox=tb,pictPen={pen & penPos={x=x+ox,y=y+oy}}}

pictdrawstring :: !String !*Picture -> *Picture
pictdrawstring string picture=:{pictContext,pictToolbox,pictPen,pictOrigin={x=ox,y=oy}}
	# (context,tb)			= WinDrawString string (pictContext,pictToolbox)
	# ((x,y),(context,tb))	= WinGetPenPos (context,tb)
	  pen					= {pictPen & penPos={x=x+ox,y=y+oy}}
	= {picture & pictContext=context,pictToolbox=tb,pictPen=pen}

pictundrawstring :: !String !*Picture -> *Picture
pictundrawstring string picture=:{pictContext,pictToolbox,pictPen=pen=:{penForeColour,penBackColour},pictOrigin={x=ox,y=oy}}
	# (context,tb)			= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)			= WinDrawString string (context,tb)
	# (context,tb)			= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	# ((x,y),(context,tb))	= WinGetPenPos (context,tb)
	= {picture & pictContext=context,pictToolbox=tb,pictPen={pen & penPos={x=x+ox,y=y+oy}}}


/*	Oval drawing operations.
	pict(draw/fill)oval center oval 
		draws/fills an oval at center with horizontal and vertical radius. The new
		pen position is not changed.
*/
pictdrawoval :: !Point !Oval !*Picture -> *Picture
pictdrawoval center oval picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinDrawOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

pictundrawoval :: !Point !Oval !*Picture -> *Picture
pictundrawoval center oval picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penBackColour,penForeColour}}
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= WinDrawOval rect (context,tb)
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

pictfilloval :: !Point !Oval !*Picture -> *Picture
pictfilloval center oval picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinFillOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

pictunfilloval :: !Point !Oval !*Picture -> *Picture
pictunfilloval center oval picture=:{pictContext,pictToolbox,pictOrigin,pictPen}
	# (context,tb)	= WinEraseOval rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect	= ovalToRect (center-pictOrigin) oval

ovalToRect :: !Point !Oval -> (!Int,!Int,!Int,!Int)
ovalToRect {x,y} {oval_rx,oval_ry}
	= (x-rx,y-ry,x+rx,y+ry)
where
	rx	= abs oval_rx
	ry	= abs oval_ry


/*	Curve drawing operations.
	pict(draw/fill)curve movePen point curve
		draws/fills a curve starting at point with a shape defined by curve. If movePen
		is True, then the new pen position is at the end of the curve, otherwise it does
		not change.
*/
pictdrawcurve :: !Bool !Point !Curve !*Picture -> *Picture
pictdrawcurve movePen start=:{x,y} curve picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinDrawCurve wrect (toTuple wstart) (toTuple wend) (pictContext,pictToolbox)
	# picture		= {picture & pictContext=context,pictToolbox=tb}
	| not movePen
		= picture
	| otherwise
		= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

pictundrawcurve :: !Bool !Point !Curve !*Picture -> *Picture
pictundrawcurve movePen start=:{x,y} curve picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penForeColour,penBackColour}}
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= WinDrawCurve wrect (toTuple wstart) (toTuple wend) (context,tb)
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	# picture		= {picture & pictContext=context,pictToolbox=tb}
	| not movePen
		= picture
	| otherwise
		= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

pictfillcurve :: !Bool !Point !Curve !*Picture -> *Picture
pictfillcurve movePen start curve picture=:{pictContext,pictToolbox,pictOrigin}
	# (context,tb)	= WinFillWedge wrect (toTuple wstart) (toTuple wend) (pictContext,pictToolbox)
	# picture		= {picture & pictContext=context,pictToolbox=tb}
	| not movePen
		= picture
	| otherwise
		= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

pictunfillcurve :: !Bool !Point !Curve !*Picture -> *Picture
pictunfillcurve movePen start curve picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penForeColour,penBackColour}}
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= WinFillWedge wrect (toTuple wstart) (toTuple wend) (context,tb)
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	# picture		= {picture & pictContext=context,pictToolbox=tb}
	| not movePen
		= picture
	| otherwise
		= setpictpenpos end picture
where
	start`				= start-pictOrigin
	(wrect,wstart,wend)	= getcurve_rect_begin_end start` curve
	end					= wend+pictOrigin

getcurve_rect_begin_end :: !Point !Curve -> (!(!Int,!Int,!Int,!Int),!Point,!Point)
getcurve_rect_begin_end start=:{x,y} {curve_oval={oval_rx,oval_ry},curve_from,curve_to,curve_clockwise}
	| curve_clockwise
		= (rect,end,start)
	| otherwise
		= (rect,start,end)
where
	rx`	= toReal (abs oval_rx)
	ry`	= toReal (abs oval_ry)
	cx	= x -(toInt ((cos curve_from)*rx`))
	cy	= y +(toInt ((sin curve_from)*ry`))
	ex	= cx+(toInt ((cos curve_to  )*rx`))
	ey	= cy-(toInt ((sin curve_to  )*ry`))
	end	= {x=ex,y=ey}
	rect= (cx-oval_rx,cy-oval_ry,cx+oval_rx,cy+oval_ry)


/*	Rect drawing operations.
	pict(draw/fill)rect rect
		draws/fills a rect. The pen position is not changed.
*/
pictdrawrect :: !(!Int,!Int,!Int,!Int) !*Picture -> *Picture
pictdrawrect (left,top,right,bottom) picture=:{pictContext,pictToolbox,pictOrigin={x=ox,y=oy}}
	# (context,tb)	= WinDrawRectangle rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect			= (left-ox,top-oy,right-ox,bottom-oy)

pictundrawrect :: !(!Int,!Int,!Int,!Int) !*Picture -> *Picture
pictundrawrect (left,top,right,bottom) picture=:{pictContext,pictToolbox,pictOrigin={x=ox,y=oy},pictPen={penForeColour,penBackColour}}
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= WinDrawRectangle rect (context,tb)
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect			= (left-ox,top-oy,right-ox,bottom-oy)

pictfillrect :: !(!Int,!Int,!Int,!Int) !*Picture -> *Picture
pictfillrect (left,top,right,bottom) picture=:{pictContext,pictToolbox,pictOrigin={x=ox,y=oy}}
	# (context,tb)	= WinFillRectangle rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect			= (left-ox,top-oy,right-ox,bottom-oy)

pictunfillrect :: !(!Int,!Int,!Int,!Int) !*Picture -> *Picture
pictunfillrect (left,top,right,bottom) picture=:{pictContext,pictToolbox,pictOrigin={x=ox,y=oy}}
	# (context,tb)	= WinEraseRectangle rect (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect			= (left-ox,top-oy,right-ox,bottom-oy)
/*
pictunfillrect (left,top,right,bottom) picture=:{pictContext,pictToolbox,pictOrigin={x=ox,y=oy},pictPen={penForeColour,penBackColour}}
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,pictToolbox)
	# (context,tb)	= WinFillRectangle rect (context,tb)
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}
where
	rect = (left-ox,top-oy,right-ox,bottom-oy)
*/

/*	Polygon drawing operations.
	pict(draw/fill)polygon point polygon
		draws/fills a polygon starting at point. The pen position is not changed.
*/
pictdrawpolygon :: !Point !Polygon !*Picture -> *Picture
pictdrawpolygon start {polygon_shape} picture=:{pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= WinDrawPolygon (pictContext,tb)
	# tb			= WinEndPolygon tb
	= {picture & pictContext=context,pictToolbox=tb}

pictundrawpolygon :: !Point !Polygon !*Picture -> *Picture
pictundrawpolygon start {polygon_shape} picture=:{pictContext,pictToolbox,pictOrigin,pictPen={penForeColour,penBackColour}}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,tb)
	# (context,tb)	= WinDrawPolygon (context,tb)
	# tb			= WinEndPolygon tb
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}

pictfillpolygon :: !Point !Polygon !*Picture -> *Picture
pictfillpolygon start {polygon_shape} picture=:{pictPen={penSize},pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= WinSetPenSize 1 (pictContext,tb)
	# (context,tb)	= WinFillPolygon (context,tb)
	# (context,tb)	= WinDrawPolygon (context,tb)
	# (context,tb)	= WinSetPenSize penSize (context,tb)
	# tb			= WinEndPolygon tb
	= {picture & pictContext=context,pictToolbox=tb}

pictunfillpolygon :: !Point !Polygon !*Picture -> *Picture
pictunfillpolygon start {polygon_shape} picture=:{pictPen={penSize,penForeColour,penBackColour},pictContext,pictToolbox,pictOrigin}
	# tb			= transferPolygon (start-pictOrigin) polygon_shape pictToolbox
	# (context,tb)	= WinSetPenColor (toRGBtriple penBackColour) (pictContext,tb)
	# (context,tb)	= WinSetPenSize 1 (context,tb)
	# (context,tb)	= WinFillPolygon  (context,tb)
	# (context,tb)	= WinDrawPolygon  (context,tb)
	# (context,tb)	= WinSetPenSize penSize (context,tb)
	# tb			= WinEndPolygon tb
	# (context,tb)	= WinSetPenColor (toRGBtriple penForeColour) (context,tb)
	= {picture & pictContext=context,pictToolbox=tb}

transferPolygon :: !Point ![Vector] !*OSToolbox -> *OSToolbox
transferPolygon start vs tb
	# tb	= WinStartPolygon (1 + length vs) tb
	# tb	= WinAddPolygonPoint wstart tb
	# tb	= transferShape wstart vs tb
	= tb
where
	wstart	= toTuple start
	
	transferShape :: !(!Int,!Int) ![Vector] !*OSToolbox -> *OSToolbox
	transferShape (x,y) [{vx,vy}:vs] tb
   		= transferShape newpos vs (WinAddPolygonPoint newpos tb)
	where
		newpos = (x+vx,y+vy)
	transferShape _ _ tb
		= tb

/*	Clipping operations.
	pictgetcliprgn gets the current clipping region.
	pictsetcliprgn sets the given clipping region.
	pictandcliprgn takes the intersection of the current clipping region and the argument region.
*/
pictgetcliprgn :: !*Picture -> (!OSRgnHandle,!*Picture)
pictgetcliprgn picture=:{pictContext,pictToolbox}
	# (cliprgn,(context,tb)) = WinGetClipRgnPicture (pictContext,pictToolbox)
	= (cliprgn,{picture & pictContext=context,pictToolbox=tb})

pictsetcliprgn :: !OSRgnHandle !*Picture -> *Picture
pictsetcliprgn cliprgn picture=:{pictContext,pictToolbox}
	# (context,tb)	= WinSetClipRgnPicture cliprgn (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}

pictandcliprgn :: !OSRgnHandle !*Picture -> *Picture
pictandcliprgn cliprgn picture=:{pictContext,pictToolbox}
	# (context,tb)	= WinClipRgnPicture cliprgn (pictContext,pictToolbox)
	= {picture & pictContext=context,pictToolbox=tb}

/*	Resolution access function (added by MW):
*/
getResolutionC :: !OSPictContext !*OSToolbox -> (!(!Int,!Int),!*OSToolbox)
getResolutionC _ _
	= code 	{
	 			ccall getResolutionC "I:VII:I"
			}

// MW: scaling of screen coordinates to printer coordinates.
getPictureScalingFactors :: !OSPictContext !*OSToolbox -> (!(!Int,!Int),!(!Int,!Int),!OSPictContext,!*OSToolbox)
getPictureScalingFactors _ _
	= code
	{
		ccall WinGetPictureScaleFactor "II-IIIIII"
	}
